package gov.va.med.mhv.sm.admin.service.impl;

import java.util.List;
import java.util.ArrayList;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.QueryParam;
import javax.validation.constraints.NotNull;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;


import gov.va.med.mhv.sm.admin.exception.SMApiException;
import gov.va.med.mhv.sm.admin.jpa.model.TriageGroup;
import gov.va.med.mhv.sm.admin.jpa.repository.TriageGroupRepository;
import gov.va.med.mhv.sm.admin.dto.TriageGroupDTO;
import gov.va.med.mhv.sm.admin.converter.TriageGroupConverter;
import gov.va.med.mhv.sm.admin.data.enums.SMErrorEnum;
import gov.va.med.mhv.sm.admin.service.util.ResponseCodeUtil;

import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ArraySchema;

@Path("/triagegroups")
@Service("triageGroupService")
@Tag(name = "Triage Group Service")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
@Component
public class TriageGroupService { 

    @Autowired
    private TriageGroupRepository triageGroupRepository;

	private static final Log log = LogFactory.getLog(TriageGroupService.class);

	private TriageGroupDTO findTriageGroupName(String name) throws SMApiException {
		
		TriageGroupDTO triageGroupDTO = null;
		TriageGroup triageGroup = null;
		try {
			if(name != null) {
				triageGroup = triageGroupRepository.findTriageGroupByName(name);
			}
			if(triageGroup != null ) {
				triageGroupDTO = TriageGroupConverter.convert(triageGroup);
			}else {
				throw new SMApiException(SMErrorEnum.TRIAGE_GROUP_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e);
			
		}
		return triageGroupDTO;
	}

	private TriageGroupDTO findActiveTriageGroupByName(String name) throws SMApiException {

		TriageGroup triageGroup = null;
		TriageGroupDTO triageGroupDTO = null;
		try {
			if (name != null) {
				triageGroup = triageGroupRepository.findActiveTriageGroupByName(name);

			}
			if (triageGroup != null) {
				triageGroupDTO = TriageGroupConverter.convert(triageGroup);
			} else {
				throw new SMApiException(SMErrorEnum.ACTIVE_TRIAGE_GROUP_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR);
		}
		return triageGroupDTO;
	}
	
	@GET
	@Operation(summary = "Get Triage Group based on Query Parameters provided.")
	@ApiResponses(value = {
	        @ApiResponse(responseCode = "200", 
	        			content = @Content(mediaType = MediaType.APPLICATION_JSON, array = @ArraySchema(schema = @Schema(implementation = TriageGroupDTO.class)))),
			@ApiResponse(responseCode = "404", description = "Triage Groups By station are not found",
						content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Unexpected Error Occurred",
						content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})	
	public  List<TriageGroupDTO> getTriageGroup(@QueryParam("name") String name,
												@QueryParam("stationNumber") String stationNumber,
												@QueryParam("active") Boolean active) throws SMApiException {

		TriageGroup triageGroup = null;
		TriageGroupDTO triageGroupDTO = null;
		List<TriageGroupDTO> triageGroupDTOList = new ArrayList<TriageGroupDTO>();
		
		try {
			
			TriageGroupEnum.SearchEnum triageGroupQueryEnum   = validateQueryParams(name, active, stationNumber);
			
			switch (triageGroupQueryEnum) {
			case  GET_TRAIGE_GROUP_BY_NAME:
				triageGroupDTO = findTriageGroupName(name);
				triageGroupDTOList.add(triageGroupDTO);
				return triageGroupDTOList;
			case  GET_ACTIVE_TRAIGE_GROUP_BY_NAME:
				triageGroupDTO = findActiveTriageGroupByName(name);
				triageGroupDTOList.add(triageGroupDTO);	
				return triageGroupDTOList;
			case GET_TRAIGE_GROUP_BY_STATION_NUMBER:
				triageGroupDTOList = getTriageGroupsByStation(stationNumber);
				return triageGroupDTOList;
			default:
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
			}
			
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR);
		}
		
	}

	
	private List<TriageGroupDTO> getTriageGroupsByStation(String stationNumber)  throws SMApiException {
		
		List<TriageGroup> triageGroupList = null;
		List<TriageGroupDTO> triageGroupDTOList = null;
		try 	{
			//Validate Station number here.
				if(stationNumber != null) {
					triageGroupList = triageGroupRepository.getTriageGroupsForStation(stationNumber);
				}
				if (triageGroupList != null) {
					triageGroupDTOList = TriageGroupConverter.convertList(triageGroupList);
				} else {
					throw new SMApiException(SMErrorEnum.TRIAGE_GROUP_NOT_FOUND_FOR_STATION.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
				}
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e);
		}
		return triageGroupDTOList;
	}

	@POST
	@Path("")
	@Operation(summary = "Create Triage Group")
	@ApiResponses(value = {
		    @ApiResponse(responseCode = "200", 
        			content = @Content(mediaType = MediaType.APPLICATION_JSON, array = @ArraySchema(schema = @Schema(implementation = TriageGroupDTO.class)))),
			@ApiResponse(responseCode = "500", description = "Internal Server Error Exception has Occurred",
					content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})	
	public TriageGroupDTO createTriageGroup(TriageGroupDTO triageGroupDTO) throws SMApiException {

		TriageGroup triageGroup = null;
		try {
			if (triageGroupDTO != null) {

				triageGroup = triageGroupRepository.save(TriageGroupConverter.convert(triageGroupDTO));
			}

			if (triageGroup != null) {
				triageGroupDTO = TriageGroupConverter.convert(triageGroup);
			}
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e);
		}
		return triageGroupDTO;
	}

	
	@PUT
	@Path("")
	@Operation(summary = "Update Triage Group")
	@ApiResponses(value = {
			@ApiResponse(responseCode = "200", 
					 	content = @Content(mediaType = MediaType.APPLICATION_JSON,array = @ArraySchema(schema = @Schema(implementation = TriageGroupDTO.class)))),
			@ApiResponse(responseCode = "404", description = "Triage Group is not found",
			 			content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class))),
			@ApiResponse(responseCode = "500", description = "Internal ServerError Exception has Occurred",
						content = @Content(mediaType = MediaType.APPLICATION_JSON, schema = @Schema(implementation = SMApiException.class)))
	})	
	public TriageGroupDTO updateTriageGroup(TriageGroupDTO triageGroupDTO) throws SMApiException {

		TriageGroup triageGroup = null;
		try {
			if (triageGroupDTO != null && triageGroupDTO.getId() !=null) {
				if(triageGroupRepository.getOne(triageGroupDTO.getId()) !=null){
					triageGroup = triageGroupRepository.save(TriageGroupConverter.convert(triageGroupDTO));
				}else{
					throw new SMApiException(SMErrorEnum.TRIAGE_GROUP_NOT_FOUND.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);		
				}
			}

			if (triageGroup != null) {
				triageGroupDTO = TriageGroupConverter.convert(triageGroup);
			}
		} catch (Exception e) {
			log.error(e);
			throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR,e);
		}
		return triageGroupDTO;
	}
	
	private static class TriageGroupEnum{
		public enum SearchEnum{
			GET_TRAIGE_GROUP_BY_NAME,
			GET_ACTIVE_TRAIGE_GROUP_BY_NAME,
			GET_TRAIGE_GROUP_BY_STATION_NUMBER;
		}
	}
	
		
	private TriageGroupEnum.SearchEnum validateQueryParams(String name, Boolean active, String stationNumber)  throws SMApiException {
		
		if(name !=null){
			if(stationNumber != null) {
				throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
			}else {
				
				if(active) {
					return TriageGroupEnum.SearchEnum.GET_ACTIVE_TRAIGE_GROUP_BY_NAME;
				}else {
					return TriageGroupEnum.SearchEnum.GET_TRAIGE_GROUP_BY_NAME;
					
				}
			}
		}else if(stationNumber != null){
			if(name !=null) {
				throw new SMApiException(SMErrorEnum.UNKNOWN_EXCEPTION.getErrorMessage(), ResponseCodeUtil.STATUS_INTERNAL_SERVER_ERROR);
			}else{
				return TriageGroupEnum.SearchEnum.GET_TRAIGE_GROUP_BY_STATION_NUMBER;
			}
		}else{
			throw new SMApiException(SMErrorEnum.INVALID_HTTP_REQUEST.getErrorMessage(), ResponseCodeUtil.STATUS_NOT_FOUND);
		}
		
	}
	
	@GET
	@Path("sayHello")	
	@Operation(summary = "Get Hello")
	@ApiResponses(value = {
//	 @ApiResponse(responseCode = "400", description = "Exception has Occured"),
//	 @ApiResponse(responseCode = "500", description = "Exception has Occured 500", 
//	 					content = @Content(mediaType = "application/json", schema = @Schema(implementation = SMApiException.class))
//	 ),
	 @ApiResponse(
             	content = @Content(mediaType = "application/json",
                schema = @Schema(implementation = String.class))
             )	  
	})
	public String sayHello()  throws SMApiException{
	//	throw new SMApiException("UserS Not Found", ResponseCodeUtil.STATUS_NOT_FOUND);
		return "Hello World";
		
	}
	
}
